"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _jsbi = _interopRequireDefault(require("jsbi"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const SHIFT_LEFT_32 = (1 << 16) * (1 << 16);
const SHIFT_RIGHT_32 = 1 / SHIFT_LEFT_32;
const UNKNOWN_PLP_LEN = Buffer.from([0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]);
const ZERO_LENGTH_BUFFER = Buffer.alloc(0);

/**
  A Buffer-like class that tracks position.

  As values are written, the position advances by the size of the written data.
  When writing, automatically allocates new buffers if there's not enough space.
 */
class WritableTrackingBuffer {
  constructor(initialSize, encoding, doubleSizeGrowth) {
    this.initialSize = void 0;
    this.encoding = void 0;
    this.doubleSizeGrowth = void 0;
    this.buffer = void 0;
    this.compositeBuffer = void 0;
    this.position = void 0;
    this.initialSize = initialSize;
    this.encoding = encoding || 'ucs2';
    this.doubleSizeGrowth = doubleSizeGrowth || false;
    this.buffer = Buffer.alloc(this.initialSize, 0);
    this.compositeBuffer = ZERO_LENGTH_BUFFER;
    this.position = 0;
  }

  get data() {
    this.newBuffer(0);
    return this.compositeBuffer;
  }

  copyFrom(buffer) {
    const length = buffer.length;
    this.makeRoomFor(length);
    buffer.copy(this.buffer, this.position);
    this.position += length;
  }

  makeRoomFor(requiredLength) {
    if (this.buffer.length - this.position < requiredLength) {
      if (this.doubleSizeGrowth) {
        let size = Math.max(128, this.buffer.length * 2);

        while (size < requiredLength) {
          size *= 2;
        }

        this.newBuffer(size);
      } else {
        this.newBuffer(requiredLength);
      }
    }
  }

  newBuffer(size) {
    const buffer = this.buffer.slice(0, this.position);
    this.compositeBuffer = Buffer.concat([this.compositeBuffer, buffer]);
    this.buffer = size === 0 ? ZERO_LENGTH_BUFFER : Buffer.alloc(size, 0);
    this.position = 0;
  }

  writeUInt8(value) {
    const length = 1;
    this.makeRoomFor(length);
    this.buffer.writeUInt8(value, this.position);
    this.position += length;
  }

  writeUInt16LE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeUInt16LE(value, this.position);
    this.position += length;
  }

  writeUShort(value) {
    this.writeUInt16LE(value);
  }

  writeUInt16BE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeUInt16BE(value, this.position);
    this.position += length;
  }

  writeUInt24LE(value) {
    const length = 3;
    this.makeRoomFor(length);
    this.buffer[this.position + 2] = value >>> 16 & 0xff;
    this.buffer[this.position + 1] = value >>> 8 & 0xff;
    this.buffer[this.position] = value & 0xff;
    this.position += length;
  }

  writeUInt32LE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeUInt32LE(value, this.position);
    this.position += length;
  }

  writeBigInt64LE(value) {
    this.writeBigU_Int64LE(value);
  }

  writeBigU_Int64LE(value) {
    this.makeRoomFor(8);

    let lo = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(value, _jsbi.default.BigInt(0xffffffff)));

    this.buffer[this.position++] = lo;
    lo = lo >> 8;
    this.buffer[this.position++] = lo;
    lo = lo >> 8;
    this.buffer[this.position++] = lo;
    lo = lo >> 8;
    this.buffer[this.position++] = lo;

    let hi = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(_jsbi.default.signedRightShift(value, _jsbi.default.BigInt(32)), _jsbi.default.BigInt(0xffffffff)));

    this.buffer[this.position++] = hi;
    hi = hi >> 8;
    this.buffer[this.position++] = hi;
    hi = hi >> 8;
    this.buffer[this.position++] = hi;
    hi = hi >> 8;
    this.buffer[this.position++] = hi;
  }

  writeInt64LE(value) {
    this.writeBigInt64LE(_jsbi.default.BigInt(value));
  }

  writeUInt32BE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeUInt32BE(value, this.position);
    this.position += length;
  }

  writeUInt40LE(value) {
    // inspired by https://github.com/dpw/node-buffer-more-ints
    this.writeInt32LE(value & -1);
    this.writeUInt8(Math.floor(value * SHIFT_RIGHT_32));
  }

  writeUInt64LE(value) {
    this.writeBigUInt64LE(_jsbi.default.BigInt(value));
  }

  writeBigUInt64LE(value) {
    this.writeBigU_Int64LE(value);
  }

  writeInt8(value) {
    const length = 1;
    this.makeRoomFor(length);
    this.buffer.writeInt8(value, this.position);
    this.position += length;
  }

  writeInt16LE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeInt16LE(value, this.position);
    this.position += length;
  }

  writeInt16BE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeInt16BE(value, this.position);
    this.position += length;
  }

  writeInt32LE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeInt32LE(value, this.position);
    this.position += length;
  }

  writeInt32BE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeInt32BE(value, this.position);
    this.position += length;
  }

  writeFloatLE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeFloatLE(value, this.position);
    this.position += length;
  }

  writeDoubleLE(value) {
    const length = 8;
    this.makeRoomFor(length);
    this.buffer.writeDoubleLE(value, this.position);
    this.position += length;
  }

  writeString(value, encoding) {
    if (encoding == null) {
      encoding = this.encoding;
    }

    const length = Buffer.byteLength(value, encoding);
    this.makeRoomFor(length); // $FlowFixMe https://github.com/facebook/flow/pull/5398

    this.buffer.write(value, this.position, encoding);
    this.position += length;
  }

  writeBVarchar(value, encoding) {
    this.writeUInt8(value.length);
    this.writeString(value, encoding);
  }

  writeUsVarchar(value, encoding) {
    this.writeUInt16LE(value.length);
    this.writeString(value, encoding);
  } // TODO: Figure out what types are passed in other than `Buffer`


  writeUsVarbyte(value, encoding) {
    if (encoding == null) {
      encoding = this.encoding;
    }

    let length;

    if (value instanceof Buffer) {
      length = value.length;
    } else {
      value = value.toString();
      length = Buffer.byteLength(value, encoding);
    }

    this.writeUInt16LE(length);

    if (value instanceof Buffer) {
      this.writeBuffer(value);
    } else {
      this.makeRoomFor(length); // $FlowFixMe https://github.com/facebook/flow/pull/5398

      this.buffer.write(value, this.position, encoding);
      this.position += length;
    }
  }

  writePLPBody(value, encoding) {
    if (encoding == null) {
      encoding = this.encoding;
    }

    let length;

    if (value instanceof Buffer) {
      length = value.length;
    } else {
      value = value.toString();
      length = Buffer.byteLength(value, encoding);
    } // Length of all chunks.
    // this.writeUInt64LE(length);
    // unknown seems to work better here - might revisit later.


    this.writeBuffer(UNKNOWN_PLP_LEN); // In the UNKNOWN_PLP_LEN case, the data is represented as a series of zero or more chunks.

    if (length > 0) {
      // One chunk.
      this.writeUInt32LE(length);

      if (value instanceof Buffer) {
        this.writeBuffer(value);
      } else {
        this.makeRoomFor(length);
        this.buffer.write(value, this.position, encoding);
        this.position += length;
      }
    } // PLP_TERMINATOR (no more chunks).


    this.writeUInt32LE(0);
  }

  writeBuffer(value) {
    const length = value.length;
    this.makeRoomFor(length);
    value.copy(this.buffer, this.position);
    this.position += length;
  }

  writeMoney(value) {
    this.writeInt32LE(Math.floor(value * SHIFT_RIGHT_32));
    this.writeInt32LE(value & -1);
  }

}

var _default = WritableTrackingBuffer;
exports.default = _default;
module.exports = WritableTrackingBuffer;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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