"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _metadataParser = require("./metadata-parser");

var _dataType = require("./data-type");

var _iconvLite = _interopRequireDefault(require("iconv-lite"));

var _sprintfJs = require("sprintf-js");

var _guidParser = require("./guid-parser");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const NULL = (1 << 16) - 1;
const MAX = (1 << 16) - 1;
const THREE_AND_A_THIRD = 3 + 1 / 3;
const MONEY_DIVISOR = 10000;
const PLP_NULL = Buffer.from([0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const UNKNOWN_PLP_LEN = Buffer.from([0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const DEFAULT_ENCODING = 'utf8';

function readTinyInt(parser, callback) {
  parser.readUInt8(callback);
}

function readSmallInt(parser, callback) {
  parser.readInt16LE(callback);
}

function readInt(parser, callback) {
  parser.readInt32LE(callback);
}

function readBigInt(parser, callback) {
  parser.readBigInt64LE(value => {
    callback(value.toString());
  });
}

function readReal(parser, callback) {
  parser.readFloatLE(callback);
}

function readFloat(parser, callback) {
  parser.readDoubleLE(callback);
}

function readSmallMoney(parser, callback) {
  parser.readInt32LE(value => {
    callback(value / MONEY_DIVISOR);
  });
}

function readMoney(parser, callback) {
  parser.readInt32LE(high => {
    parser.readUInt32LE(low => {
      callback((low + 0x100000000 * high) / MONEY_DIVISOR);
    });
  });
}

function readBit(parser, callback) {
  parser.readUInt8(value => {
    callback(!!value);
  });
}

function valueParse(parser, metadata, options, callback) {
  const type = metadata.type;

  switch (type.name) {
    case 'Null':
      return callback(null);

    case 'TinyInt':
      return readTinyInt(parser, callback);

    case 'SmallInt':
      return readSmallInt(parser, callback);

    case 'Int':
      return readInt(parser, callback);

    case 'BigInt':
      return readBigInt(parser, callback);

    case 'IntN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 1:
            return readTinyInt(parser, callback);

          case 2:
            return readSmallInt(parser, callback);

          case 4:
            return readInt(parser, callback);

          case 8:
            return readBigInt(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for IntN');
        }
      });

    case 'Real':
      return readReal(parser, callback);

    case 'Float':
      return readFloat(parser, callback);

    case 'FloatN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 4:
            return readReal(parser, callback);

          case 8:
            return readFloat(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for FloatN');
        }
      });

    case 'SmallMoney':
      return readSmallMoney(parser, callback);

    case 'Money':
      return readMoney(parser, callback);

    case 'MoneyN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 4:
            return readSmallMoney(parser, callback);

          case 8:
            return readMoney(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for MoneyN');
        }
      });

    case 'Bit':
      return readBit(parser, callback);

    case 'BitN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 1:
            return readBit(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for BitN');
        }
      });

    case 'VarChar':
    case 'Char':
      const codepage = metadata.collation.codepage;

      if (metadata.dataLength === MAX) {
        return readMaxChars(parser, codepage, callback);
      } else {
        return parser.readUInt16LE(dataLength => {
          if (dataLength === NULL) {
            return callback(null);
          }

          readChars(parser, dataLength, codepage, callback);
        });
      }

    case 'NVarChar':
    case 'NChar':
      if (metadata.dataLength === MAX) {
        return readMaxNChars(parser, callback);
      } else {
        return parser.readUInt16LE(dataLength => {
          if (dataLength === NULL) {
            return callback(null);
          }

          readNChars(parser, dataLength, callback);
        });
      }

    case 'VarBinary':
    case 'Binary':
      if (metadata.dataLength === MAX) {
        return readMaxBinary(parser, callback);
      } else {
        return parser.readUInt16LE(dataLength => {
          if (dataLength === NULL) {
            return callback(null);
          }

          readBinary(parser, dataLength, callback);
        });
      }

    case 'Text':
      return parser.readUInt8(textPointerLength => {
        if (textPointerLength === 0) {
          return callback(null);
        }

        parser.readBuffer(textPointerLength, _textPointer => {
          parser.readBuffer(8, _timestamp => {
            parser.readUInt32LE(dataLength => {
              readChars(parser, dataLength, metadata.collation.codepage, callback);
            });
          });
        });
      });

    case 'NText':
      return parser.readUInt8(textPointerLength => {
        if (textPointerLength === 0) {
          return callback(null);
        }

        parser.readBuffer(textPointerLength, _textPointer => {
          parser.readBuffer(8, _timestamp => {
            parser.readUInt32LE(dataLength => {
              readNChars(parser, dataLength, callback);
            });
          });
        });
      });

    case 'Image':
      return parser.readUInt8(textPointerLength => {
        if (textPointerLength === 0) {
          return callback(null);
        }

        parser.readBuffer(textPointerLength, _textPointer => {
          parser.readBuffer(8, _timestamp => {
            parser.readUInt32LE(dataLength => {
              readBinary(parser, dataLength, callback);
            });
          });
        });
      });

    case 'Xml':
      return readMaxNChars(parser, callback);

    case 'SmallDateTime':
      return readSmallDateTime(parser, options.useUTC, callback);

    case 'DateTime':
      return readDateTime(parser, options.useUTC, callback);

    case 'DateTimeN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 4:
            return readSmallDateTime(parser, options.useUTC, callback);

          case 8:
            return readDateTime(parser, options.useUTC, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for DateTimeN');
        }
      });

    case 'Time':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readTime(parser, dataLength, metadata.scale, options.useUTC, callback);
        }
      });

    case 'Date':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readDate(parser, options.useUTC, callback);
        }
      });

    case 'DateTime2':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readDateTime2(parser, dataLength, metadata.scale, options.useUTC, callback);
        }
      });

    case 'DateTimeOffset':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readDateTimeOffset(parser, dataLength, metadata.scale, callback);
        }
      });

    case 'NumericN':
    case 'DecimalN':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readNumeric(parser, dataLength, metadata.precision, metadata.scale, callback);
        }
      });

    case 'UniqueIdentifier':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 0x10:
            return readUniqueIdentifier(parser, options, callback);

          default:
            throw new Error((0, _sprintfJs.sprintf)('Unsupported guid size %d', dataLength - 1));
        }
      });

    case 'UDT':
      return readMaxBinary(parser, callback);

    case 'Variant':
      return parser.readUInt32LE(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        }

        readVariant(parser, options, dataLength, callback);
      });

    default:
      throw new Error((0, _sprintfJs.sprintf)('Unrecognised type %s', type.name));
  }
}

function readUniqueIdentifier(parser, options, callback) {
  parser.readBuffer(0x10, data => {
    callback(options.lowerCaseGuids ? (0, _guidParser.bufferToLowerCaseGuid)(data) : (0, _guidParser.bufferToUpperCaseGuid)(data));
  });
}

function readNumeric(parser, dataLength, _precision, scale, callback) {
  parser.readUInt8(sign => {
    sign = sign === 1 ? 1 : -1;
    let readValue;

    if (dataLength === 5) {
      readValue = parser.readUInt32LE;
    } else if (dataLength === 9) {
      readValue = parser.readUNumeric64LE;
    } else if (dataLength === 13) {
      readValue = parser.readUNumeric96LE;
    } else if (dataLength === 17) {
      readValue = parser.readUNumeric128LE;
    } else {
      throw new Error((0, _sprintfJs.sprintf)('Unsupported numeric dataLength %d', dataLength));
    }

    readValue.call(parser, value => {
      callback(value * sign / Math.pow(10, scale));
    });
  });
}

function readVariant(parser, options, dataLength, callback) {
  return parser.readUInt8(baseType => {
    const type = _dataType.TYPE[baseType];
    return parser.readUInt8(propBytes => {
      dataLength = dataLength - propBytes - 2;

      switch (type.name) {
        case 'UniqueIdentifier':
          return readUniqueIdentifier(parser, options, callback);

        case 'Bit':
          return readBit(parser, callback);

        case 'TinyInt':
          return readTinyInt(parser, callback);

        case 'SmallInt':
          return readSmallInt(parser, callback);

        case 'Int':
          return readInt(parser, callback);

        case 'BigInt':
          return readBigInt(parser, callback);

        case 'SmallDateTime':
          return readSmallDateTime(parser, options.useUTC, callback);

        case 'DateTime':
          return readDateTime(parser, options.useUTC, callback);

        case 'Real':
          return readReal(parser, callback);

        case 'Float':
          return readFloat(parser, callback);

        case 'SmallMoney':
          return readSmallMoney(parser, callback);

        case 'Money':
          return readMoney(parser, callback);

        case 'Date':
          return readDate(parser, options.useUTC, callback);

        case 'Time':
          return parser.readUInt8(scale => {
            return readTime(parser, dataLength, scale, options.useUTC, callback);
          });

        case 'DateTime2':
          return parser.readUInt8(scale => {
            return readDateTime2(parser, dataLength, scale, options.useUTC, callback);
          });

        case 'DateTimeOffset':
          return parser.readUInt8(scale => {
            return readDateTimeOffset(parser, dataLength, scale, callback);
          });

        case 'VarBinary':
        case 'Binary':
          return parser.readUInt16LE(_maxLength => {
            readBinary(parser, dataLength, callback);
          });

        case 'NumericN':
        case 'DecimalN':
          return parser.readUInt8(precision => {
            parser.readUInt8(scale => {
              readNumeric(parser, dataLength, precision, scale, callback);
            });
          });

        case 'VarChar':
        case 'Char':
          return parser.readUInt16LE(_maxLength => {
            (0, _metadataParser.readCollation)(parser, collation => {
              readChars(parser, dataLength, collation.codepage, callback);
            });
          });

        case 'NVarChar':
        case 'NChar':
          return parser.readUInt16LE(_maxLength => {
            (0, _metadataParser.readCollation)(parser, _collation => {
              readNChars(parser, dataLength, callback);
            });
          });

        default:
          throw new Error('Invalid type!');
      }
    });
  });
}

function readBinary(parser, dataLength, callback) {
  return parser.readBuffer(dataLength, callback);
}

function readChars(parser, dataLength, codepage, callback) {
  if (codepage == null) {
    codepage = DEFAULT_ENCODING;
  }

  return parser.readBuffer(dataLength, data => {
    callback(_iconvLite.default.decode(data, codepage));
  });
}

function readNChars(parser, dataLength, callback) {
  parser.readBuffer(dataLength, data => {
    callback(data.toString('ucs2'));
  });
}

function readMaxBinary(parser, callback) {
  return readMax(parser, callback);
}

function readMaxChars(parser, codepage, callback) {
  if (codepage == null) {
    codepage = DEFAULT_ENCODING;
  }

  readMax(parser, data => {
    if (data) {
      callback(_iconvLite.default.decode(data, codepage));
    } else {
      callback(null);
    }
  });
}

function readMaxNChars(parser, callback) {
  readMax(parser, data => {
    if (data) {
      callback(data.toString('ucs2'));
    } else {
      callback(null);
    }
  });
}

function readMax(parser, callback) {
  parser.readBuffer(8, type => {
    if (type.equals(PLP_NULL)) {
      return callback(null);
    } else if (type.equals(UNKNOWN_PLP_LEN)) {
      return readMaxUnknownLength(parser, callback);
    } else {
      const low = type.readUInt32LE(0);
      const high = type.readUInt32LE(4);

      if (high >= 2 << 53 - 32) {
        console.warn('Read UInt64LE > 53 bits : high=' + high + ', low=' + low);
      }

      const expectedLength = low + 0x100000000 * high;
      return readMaxKnownLength(parser, expectedLength, callback);
    }
  });
}

function readMaxKnownLength(parser, totalLength, callback) {
  const data = Buffer.alloc(totalLength, 0);
  let offset = 0;

  function next(done) {
    parser.readUInt32LE(chunkLength => {
      if (!chunkLength) {
        return done();
      }

      parser.readBuffer(chunkLength, chunk => {
        chunk.copy(data, offset);
        offset += chunkLength;
        next(done);
      });
    });
  }

  next(() => {
    if (offset !== totalLength) {
      throw new Error('Partially Length-prefixed Bytes unmatched lengths : expected ' + totalLength + ', but got ' + offset + ' bytes');
    }

    callback(data);
  });
}

function readMaxUnknownLength(parser, callback) {
  const chunks = [];
  let length = 0;

  function next(done) {
    parser.readUInt32LE(chunkLength => {
      if (!chunkLength) {
        return done();
      }

      parser.readBuffer(chunkLength, chunk => {
        chunks.push(chunk);
        length += chunkLength;
        next(done);
      });
    });
  }

  next(() => {
    callback(Buffer.concat(chunks, length));
  });
}

function readSmallDateTime(parser, useUTC, callback) {
  parser.readUInt16LE(days => {
    parser.readUInt16LE(minutes => {
      let value;

      if (useUTC) {
        value = new Date(Date.UTC(1900, 0, 1 + days, 0, minutes));
      } else {
        value = new Date(1900, 0, 1 + days, 0, minutes);
      }

      callback(value);
    });
  });
}

function readDateTime(parser, useUTC, callback) {
  parser.readInt32LE(days => {
    parser.readUInt32LE(threeHundredthsOfSecond => {
      const milliseconds = Math.round(threeHundredthsOfSecond * THREE_AND_A_THIRD);
      let value;

      if (useUTC) {
        value = new Date(Date.UTC(1900, 0, 1 + days, 0, 0, 0, milliseconds));
      } else {
        value = new Date(1900, 0, 1 + days, 0, 0, 0, milliseconds);
      }

      callback(value);
    });
  });
}

function readTime(parser, dataLength, scale, useUTC, callback) {
  let readValue;

  switch (dataLength) {
    case 3:
      readValue = parser.readUInt24LE;
      break;

    case 4:
      readValue = parser.readUInt32LE;
      break;

    case 5:
      readValue = parser.readUInt40LE;
  }

  readValue.call(parser, value => {
    if (scale < 7) {
      for (let i = scale; i < 7; i++) {
        value *= 10;
      }
    }

    let date;

    if (useUTC) {
      date = new Date(Date.UTC(1970, 0, 1, 0, 0, 0, value / 10000));
    } else {
      date = new Date(1970, 0, 1, 0, 0, 0, value / 10000);
    }

    Object.defineProperty(date, 'nanosecondsDelta', {
      enumerable: false,
      value: value % 10000 / Math.pow(10, 7)
    });
    callback(date);
  });
}

function readDate(parser, useUTC, callback) {
  parser.readUInt24LE(days => {
    if (useUTC) {
      callback(new Date(Date.UTC(2000, 0, days - 730118)));
    } else {
      callback(new Date(2000, 0, days - 730118));
    }
  });
}

function readDateTime2(parser, dataLength, scale, useUTC, callback) {
  readTime(parser, dataLength - 3, scale, useUTC, time => {
    // TODO: 'input' is 'time', but TypeScript cannot find "time.nanosecondsDelta";
    parser.readUInt24LE(days => {
      let date;

      if (useUTC) {
        date = new Date(Date.UTC(2000, 0, days - 730118, 0, 0, 0, +time));
      } else {
        date = new Date(2000, 0, days - 730118, time.getHours(), time.getMinutes(), time.getSeconds(), time.getMilliseconds());
      }

      Object.defineProperty(date, 'nanosecondsDelta', {
        enumerable: false,
        value: time.nanosecondsDelta
      });
      callback(date);
    });
  });
}

function readDateTimeOffset(parser, dataLength, scale, callback) {
  readTime(parser, dataLength - 5, scale, true, time => {
    parser.readUInt24LE(days => {
      // offset
      parser.readInt16LE(() => {
        const date = new Date(Date.UTC(2000, 0, days - 730118, 0, 0, 0, +time));
        Object.defineProperty(date, 'nanosecondsDelta', {
          enumerable: false,
          value: time.nanosecondsDelta
        });
        callback(date);
      });
    });
  });
}

var _default = valueParse;
exports.default = _default;
module.exports = valueParse;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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