import $Refs from "./refs.js";
import type { JSONSchema } from "./types/index.js";
interface ResolvedInput {
    path: string;
    schema: string | JSONSchema | Buffer | Awaited<JSONSchema> | undefined;
    type: "file" | "json" | "url";
}
export declare const getResolvedInput: ({ pathOrUrlOrSchema, }: {
    pathOrUrlOrSchema: JSONSchema | string | unknown;
}) => ResolvedInput;
/**
 * This class parses a JSON schema, builds a map of its JSON references and their resolved values,
 * and provides methods for traversing, manipulating, and dereferencing those references.
 */
export declare class $RefParser {
    /**
     * The resolved JSON references
     *
     * @type {$Refs}
     * @readonly
     */
    $refs: $Refs<JSONSchema>;
    options: import("./options.js").$RefParserOptions;
    /**
     * The parsed (and possibly dereferenced) JSON schema object
     *
     * @type {object}
     * @readonly
     */
    schema: JSONSchema | null;
    schemaMany: JSONSchema[];
    schemaManySources: string[];
    sourcePathToPrefix: Map<string, string>;
    /**
     * Bundles all referenced files/URLs into a single schema that only has internal `$ref` pointers. This lets you split-up your schema however you want while you're building it, but easily combine all those files together when it's time to package or distribute the schema to other people. The resulting schema size will be small, since it will still contain internal JSON references rather than being fully-dereferenced.
     *
     * This also eliminates the risk of circular references, so the schema can be safely serialized using `JSON.stringify()`.
     *
     * See https://apitools.dev/json-schema-ref-parser/docs/ref-parser.html#bundleschema-options-callback
     *
     * @param pathOrUrlOrSchema A JSON Schema object, or the file path or URL of a JSON Schema file.
     */
    bundle({ arrayBuffer, fetch, pathOrUrlOrSchema, resolvedInput, }: {
        arrayBuffer?: ArrayBuffer;
        fetch?: RequestInit;
        pathOrUrlOrSchema: JSONSchema | string | unknown;
        resolvedInput?: ResolvedInput;
    }): Promise<JSONSchema>;
    /**
     * Bundles multiple roots (files/URLs/objects) into a single schema by creating a synthetic root
     * that references each input, resolving all externals, and then hoisting via the existing bundler.
     */
    bundleMany({ arrayBuffer, fetch, pathOrUrlOrSchemas, resolvedInputs, }: {
        arrayBuffer?: ArrayBuffer[];
        fetch?: RequestInit;
        pathOrUrlOrSchemas: Array<JSONSchema | string | unknown>;
        resolvedInputs?: ResolvedInput[];
    }): Promise<JSONSchema>;
    /**
     * Dereferences all `$ref` pointers in the JSON Schema, replacing each reference with its resolved value. This results in a schema object that does not contain any `$ref` pointers. Instead, it's a normal JavaScript object tree that can easily be crawled and used just like any other JavaScript object. This is great for programmatic usage, especially when using tools that don't understand JSON references.
     *
     * The dereference method maintains object reference equality, meaning that all `$ref` pointers that point to the same object will be replaced with references to the same object. Again, this is great for programmatic usage, but it does introduce the risk of circular references, so be careful if you intend to serialize the schema using `JSON.stringify()`. Consider using the bundle method instead, which does not create circular references.
     *
     * See https://apitools.dev/json-schema-ref-parser/docs/ref-parser.html#dereferenceschema-options-callback
     *
     * @param pathOrUrlOrSchema A JSON Schema object, or the file path or URL of a JSON Schema file.
     */
    dereference({ fetch, pathOrUrlOrSchema, }: {
        fetch?: RequestInit;
        pathOrUrlOrSchema: JSONSchema | string | unknown;
    }): Promise<JSONSchema>;
    /**
     * Parses the given JSON schema.
     * This method does not resolve any JSON references.
     * It just reads a single file in JSON or YAML format, and parse it as a JavaScript object.
     *
     * @param pathOrUrlOrSchema A JSON Schema object, or the file path or URL of a JSON Schema file.
     * @returns - The returned promise resolves with the parsed JSON schema object.
     */
    parse({ arrayBuffer, fetch, pathOrUrlOrSchema, resolvedInput: _resolvedInput, }: {
        arrayBuffer?: ArrayBuffer;
        fetch?: RequestInit;
        pathOrUrlOrSchema: JSONSchema | string | unknown;
        resolvedInput?: ResolvedInput;
    }): Promise<{
        schema: JSONSchema;
    }>;
    private parseMany;
    mergeMany(): JSONSchema;
}
export { sendRequest } from "./resolvers/url.js";
export type { JSONSchema } from "./types/index.js";
