"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TYPE = exports.Packet = exports.OFFSET = exports.HEADER_LENGTH = void 0;
exports.isPacketComplete = isPacketComplete;
exports.packetLength = packetLength;
var _sprintfJs = require("sprintf-js");
const HEADER_LENGTH = exports.HEADER_LENGTH = 8;
const TYPE = exports.TYPE = {
  SQL_BATCH: 0x01,
  RPC_REQUEST: 0x03,
  TABULAR_RESULT: 0x04,
  ATTENTION: 0x06,
  BULK_LOAD: 0x07,
  TRANSACTION_MANAGER: 0x0E,
  LOGIN7: 0x10,
  NTLMAUTH_PKT: 0x11,
  PRELOGIN: 0x12,
  FEDAUTH_TOKEN: 0x08
};
const typeByValue = {};
for (const name in TYPE) {
  typeByValue[TYPE[name]] = name;
}
const STATUS = {
  NORMAL: 0x00,
  EOM: 0x01,
  IGNORE: 0x02,
  RESETCONNECTION: 0x08,
  RESETCONNECTIONSKIPTRAN: 0x10
};
const OFFSET = exports.OFFSET = {
  Type: 0,
  Status: 1,
  Length: 2,
  SPID: 4,
  PacketID: 6,
  Window: 7
};
const DEFAULT_SPID = 0;
const DEFAULT_PACKETID = 1;
const DEFAULT_WINDOW = 0;
const NL = '\n';
class Packet {
  constructor(typeOrBuffer) {
    if (typeOrBuffer instanceof Buffer) {
      this.buffer = typeOrBuffer;
    } else {
      const type = typeOrBuffer;
      this.buffer = Buffer.alloc(HEADER_LENGTH, 0);
      this.buffer.writeUInt8(type, OFFSET.Type);
      this.buffer.writeUInt8(STATUS.NORMAL, OFFSET.Status);
      this.buffer.writeUInt16BE(DEFAULT_SPID, OFFSET.SPID);
      this.buffer.writeUInt8(DEFAULT_PACKETID, OFFSET.PacketID);
      this.buffer.writeUInt8(DEFAULT_WINDOW, OFFSET.Window);
      this.setLength();
    }
  }
  setLength() {
    this.buffer.writeUInt16BE(this.buffer.length, OFFSET.Length);
  }
  length() {
    return this.buffer.readUInt16BE(OFFSET.Length);
  }
  resetConnection(reset) {
    let status = this.buffer.readUInt8(OFFSET.Status);
    if (reset) {
      status |= STATUS.RESETCONNECTION;
    } else {
      status &= 0xFF - STATUS.RESETCONNECTION;
    }
    this.buffer.writeUInt8(status, OFFSET.Status);
  }
  last(last) {
    let status = this.buffer.readUInt8(OFFSET.Status);
    if (arguments.length > 0) {
      if (last) {
        status |= STATUS.EOM;
      } else {
        status &= 0xFF - STATUS.EOM;
      }
      this.buffer.writeUInt8(status, OFFSET.Status);
    }
    return this.isLast();
  }
  ignore(last) {
    let status = this.buffer.readUInt8(OFFSET.Status);
    if (last) {
      status |= STATUS.IGNORE;
    } else {
      status &= 0xFF - STATUS.IGNORE;
    }
    this.buffer.writeUInt8(status, OFFSET.Status);
  }
  isLast() {
    return !!(this.buffer.readUInt8(OFFSET.Status) & STATUS.EOM);
  }
  packetId(packetId) {
    if (packetId) {
      this.buffer.writeUInt8(packetId % 256, OFFSET.PacketID);
    }
    return this.buffer.readUInt8(OFFSET.PacketID);
  }
  addData(data) {
    this.buffer = Buffer.concat([this.buffer, data]);
    this.setLength();
    return this;
  }
  data() {
    return this.buffer.slice(HEADER_LENGTH);
  }
  type() {
    return this.buffer.readUInt8(OFFSET.Type);
  }
  statusAsString() {
    const status = this.buffer.readUInt8(OFFSET.Status);
    const statuses = [];
    for (const name in STATUS) {
      const value = STATUS[name];
      if (status & value) {
        statuses.push(name);
      } else {
        statuses.push(undefined);
      }
    }
    return statuses.join(' ').trim();
  }
  headerToString(indent = '') {
    const text = (0, _sprintfJs.sprintf)('type:0x%02X(%s), status:0x%02X(%s), length:0x%04X, spid:0x%04X, packetId:0x%02X, window:0x%02X', this.buffer.readUInt8(OFFSET.Type), typeByValue[this.buffer.readUInt8(OFFSET.Type)], this.buffer.readUInt8(OFFSET.Status), this.statusAsString(), this.buffer.readUInt16BE(OFFSET.Length), this.buffer.readUInt16BE(OFFSET.SPID), this.buffer.readUInt8(OFFSET.PacketID), this.buffer.readUInt8(OFFSET.Window));
    return indent + text;
  }
  dataToString(indent = '') {
    const BYTES_PER_GROUP = 0x04;
    const CHARS_PER_GROUP = 0x08;
    const BYTES_PER_LINE = 0x20;
    const data = this.data();
    let dataDump = '';
    let chars = '';
    for (let offset = 0; offset < data.length; offset++) {
      if (offset % BYTES_PER_LINE === 0) {
        dataDump += indent;
        dataDump += (0, _sprintfJs.sprintf)('%04X  ', offset);
      }
      if (data[offset] < 0x20 || data[offset] > 0x7E) {
        chars += '.';
        if ((offset + 1) % CHARS_PER_GROUP === 0 && !((offset + 1) % BYTES_PER_LINE === 0)) {
          chars += ' ';
        }
      } else {
        chars += String.fromCharCode(data[offset]);
      }
      if (data[offset] != null) {
        dataDump += (0, _sprintfJs.sprintf)('%02X', data[offset]);
      }
      if ((offset + 1) % BYTES_PER_GROUP === 0 && !((offset + 1) % BYTES_PER_LINE === 0)) {
        dataDump += ' ';
      }
      if ((offset + 1) % BYTES_PER_LINE === 0) {
        dataDump += '  ' + chars;
        chars = '';
        if (offset < data.length - 1) {
          dataDump += NL;
        }
      }
    }
    if (chars.length) {
      dataDump += '  ' + chars;
    }
    return dataDump;
  }
  toString(indent = '') {
    return this.headerToString(indent) + '\n' + this.dataToString(indent + indent);
  }
  payloadString() {
    return '';
  }
}
exports.Packet = Packet;
function isPacketComplete(potentialPacketBuffer) {
  if (potentialPacketBuffer.length < HEADER_LENGTH) {
    return false;
  } else {
    return potentialPacketBuffer.length >= potentialPacketBuffer.readUInt16BE(OFFSET.Length);
  }
}
function packetLength(potentialPacketBuffer) {
  return potentialPacketBuffer.readUInt16BE(OFFSET.Length);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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