"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _sprintfJs = require("sprintf-js");
var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));
var _crypto = require("crypto");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
const optionBufferSize = 20;
const traceIdSize = 36;
const TOKEN = {
  VERSION: 0x00,
  ENCRYPTION: 0x01,
  INSTOPT: 0x02,
  THREADID: 0x03,
  MARS: 0x04,
  TRACEID: 0x05,
  FEDAUTHREQUIRED: 0x06,
  TERMINATOR: 0xFF
};
const ENCRYPT = {
  OFF: 0x00,
  ON: 0x01,
  NOT_SUP: 0x02,
  REQ: 0x03
};
const encryptByValue = {};
for (const name in ENCRYPT) {
  const value = ENCRYPT[name];
  encryptByValue[value] = name;
}
const MARS = {
  OFF: 0x00,
  ON: 0x01
};
const marsByValue = {};
for (const name in MARS) {
  const value = MARS[name];
  marsByValue[value] = name;
}
/*
  s2.2.6.4
 */
class PreloginPayload {
  constructor(bufferOrOptions = {
    encrypt: false,
    version: {
      major: 0,
      minor: 0,
      build: 0,
      subbuild: 0
    }
  }) {
    if (bufferOrOptions instanceof Buffer) {
      this.data = bufferOrOptions;
      this.options = {
        encrypt: false,
        version: {
          major: 0,
          minor: 0,
          build: 0,
          subbuild: 0
        }
      };
    } else {
      this.options = bufferOrOptions;
      this.createOptions();
    }
    this.extractOptions();
  }
  createOptions() {
    const options = [this.createVersionOption(), this.createEncryptionOption(), this.createInstanceOption(), this.createThreadIdOption(), this.createMarsOption(), this.createTraceIdOption(), this.createFedAuthOption()];
    let length = 0;
    for (let i = 0, len = options.length; i < len; i++) {
      const option = options[i];
      length += 5 + option.data.length;
    }
    length++; // terminator
    this.data = Buffer.alloc(length, 0);
    let optionOffset = 0;
    let optionDataOffset = 5 * options.length + 1;
    for (let j = 0, len = options.length; j < len; j++) {
      const option = options[j];
      this.data.writeUInt8(option.token, optionOffset + 0);
      this.data.writeUInt16BE(optionDataOffset, optionOffset + 1);
      this.data.writeUInt16BE(option.data.length, optionOffset + 3);
      optionOffset += 5;
      option.data.copy(this.data, optionDataOffset);
      optionDataOffset += option.data.length;
    }
    this.data.writeUInt8(TOKEN.TERMINATOR, optionOffset);
  }
  createVersionOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(this.options.version.major);
    buffer.writeUInt8(this.options.version.minor);
    buffer.writeUInt16BE(this.options.version.build);
    buffer.writeUInt16BE(this.options.version.subbuild);
    return {
      token: TOKEN.VERSION,
      data: buffer.data
    };
  }
  createEncryptionOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    if (this.options.encrypt) {
      buffer.writeUInt8(ENCRYPT.ON);
    } else {
      buffer.writeUInt8(ENCRYPT.NOT_SUP);
    }
    return {
      token: TOKEN.ENCRYPTION,
      data: buffer.data
    };
  }
  createInstanceOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(0x00);
    return {
      token: TOKEN.INSTOPT,
      data: buffer.data
    };
  }
  createThreadIdOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt32BE(0x00);
    return {
      token: TOKEN.THREADID,
      data: buffer.data
    };
  }
  createMarsOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(MARS.OFF);
    return {
      token: TOKEN.MARS,
      data: buffer.data
    };
  }
  createTraceIdOption() {
    const buffer = new _writableTrackingBuffer.default(traceIdSize);
    // Generate a random series of bytes to use as the TraceID.
    // Used for debugging purposes.
    buffer.writeBuffer((0, _crypto.randomBytes)(traceIdSize));
    return {
      token: TOKEN.TRACEID,
      data: buffer.data
    };
  }
  createFedAuthOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(0x01);
    return {
      token: TOKEN.FEDAUTHREQUIRED,
      data: buffer.data
    };
  }
  extractOptions() {
    let offset = 0;
    while (this.data[offset] !== TOKEN.TERMINATOR) {
      let dataOffset = this.data.readUInt16BE(offset + 1);
      const dataLength = this.data.readUInt16BE(offset + 3);
      switch (this.data[offset]) {
        case TOKEN.VERSION:
          this.extractVersion(dataOffset);
          break;
        case TOKEN.ENCRYPTION:
          this.extractEncryption(dataOffset);
          break;
        case TOKEN.INSTOPT:
          this.extractInstance(dataOffset);
          break;
        case TOKEN.THREADID:
          if (dataLength > 0) {
            this.extractThreadId(dataOffset);
          }
          break;
        case TOKEN.MARS:
          this.extractMars(dataOffset);
          break;
        case TOKEN.TRACEID:
          this.extractTraceId(dataOffset);
          break;
        case TOKEN.FEDAUTHREQUIRED:
          this.extractFedAuth(dataOffset);
          break;
      }
      offset += 5;
      dataOffset += dataLength;
    }
  }
  extractVersion(offset) {
    this.version = {
      major: this.data.readUInt8(offset + 0),
      minor: this.data.readUInt8(offset + 1),
      build: this.data.readUInt16BE(offset + 2),
      subbuild: this.data.readUInt16BE(offset + 4)
    };
  }
  extractEncryption(offset) {
    this.encryption = this.data.readUInt8(offset);
    this.encryptionString = encryptByValue[this.encryption];
  }
  extractInstance(offset) {
    this.instance = this.data.readUInt8(offset);
  }
  extractThreadId(offset) {
    this.threadId = this.data.readUInt32BE(offset);
  }
  extractMars(offset) {
    this.mars = this.data.readUInt8(offset);
    this.marsString = marsByValue[this.mars];
  }
  extractTraceId(offset) {
    this.traceId = this.data.subarray(offset, offset + traceIdSize);
  }
  extractFedAuth(offset) {
    this.fedAuthRequired = this.data.readUInt8(offset);
  }
  toString(indent = '') {
    return indent + 'PreLogin - ' + (0, _sprintfJs.sprintf)('version:%d.%d.%d.%d, encryption:0x%02X(%s), instopt:0x%02X, threadId:0x%08X, mars:0x%02X(%s), traceId:%s', this.version.major, this.version.minor, this.version.build, this.version.subbuild, this.encryption ? this.encryption : 0, this.encryptionString ? this.encryptionString : '', this.instance ? this.instance : 0, this.threadId ? this.threadId : 0, this.mars ? this.mars : 0, this.marsString ? this.marsString : '', this.traceId ? this.traceId.toString('hex') : '');
  }
}
var _default = exports.default = PreloginPayload;
module.exports = PreloginPayload;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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