import { GraphQLResolveInfo } from 'graphql';
import { SchemaDirectiveVisitor } from 'graphql-tools';
import { IRateLimiterOptions, RateLimiterAbstract, RateLimiterRes } from 'rate-limiter-flexible';
/**
 * Configure rate limit field behaviour.
 */
export interface RateLimitArgs {
    /**
     * Number of occurrences allowed over duration.
     */
    limit: number;
    /**
     * Number of seconds before limit is reset.
     */
    duration: number;
}
export declare type RateLimitKeyGenerator<TContext> = (directiveArgs: RateLimitArgs, obj: any, args: {
    [key: string]: any;
}, context: TContext, info: GraphQLResolveInfo) => Promise<string> | string;
export declare type RateLimitPointsCalculator<TContext> = (directiveArgs: RateLimitArgs, obj: any, args: {
    [key: string]: any;
}, context: TContext, info: GraphQLResolveInfo) => Promise<number> | number;
export declare type RateLimitOnLimit<TContext> = (resource: RateLimiterRes, directiveArgs: RateLimitArgs, obj: any, args: {
    [key: string]: any;
}, context: TContext, info: GraphQLResolveInfo) => any;
/**
 * Configure rate limit behaviour.
 */
export interface IOptions<TContext> {
    /**
     * Constructs a key to represent an operation on a field.
     */
    keyGenerator?: RateLimitKeyGenerator<TContext>;
    /**
     * Calculate the number of points to consume.
     */
    pointsCalculator?: RateLimitPointsCalculator<TContext>;
    /**
     * Behaviour when limit is exceeded.
     */
    onLimit?: RateLimitOnLimit<TContext>;
    /**
     * An implementation of a limiter.
     */
    limiterClass?: typeof RateLimiterAbstract;
    /**
     * Configuration to apply to created limiters.
     */
    limiterOptions?: Pick<IRateLimiterOptions, Exclude<keyof IRateLimiterOptions, keyof {
        points?: number;
        duration?: number;
    }>>;
}
/**
 * Get a value to uniquely identify a field in a schema.
 * @param directiveArgs The arguments defined in the schema for the directive.
 * @param obj The previous result returned from the resolver on the parent field.
 * @param args The arguments provided to the field in the GraphQL operation.
 * @param context Contains per-request state shared by all resolvers in a particular operation.
 * @param info Holds field-specific information relevant to the current operation as well as the schema details.
 */
export declare function defaultKeyGenerator<TContext>(directiveArgs: RateLimitArgs, obj: any, args: {
    [key: string]: any;
}, context: TContext, info: GraphQLResolveInfo): string;
/**
 * Calculate the number of points to consume.
 * @param directiveArgs The arguments defined in the schema for the directive.
 * @param obj The previous result returned from the resolver on the parent field.
 * @param args The arguments provided to the field in the GraphQL operation.
 * @param context Contains per-request state shared by all resolvers in a particular operation.
 * @param info Holds field-specific information relevant to the current operation as well as the schema details.
 */
export declare function defaultPointsCalculator<TContext>(directiveArgs: RateLimitArgs, obj: any, args: {
    [key: string]: any;
}, context: TContext, info: GraphQLResolveInfo): number;
/**
 * Raise a rate limit error when there are too many requests.
 * @param resource The current rate limit information for this field.
 * @param directiveArgs The arguments defined in the schema for the directive.
 * @param obj The previous result returned from the resolver on the parent field.
 * @param args The arguments provided to the field in the GraphQL operation.
 * @param context Contains per-request state shared by all resolvers in a particular operation.
 * @param info Holds field-specific information relevant to the current operation as well as the schema details.
 */
export declare function defaultOnLimit<TContext>(resource: RateLimiterRes, directiveArgs: RateLimitArgs, obj: any, args: {
    [key: string]: any;
}, context: TContext, info: GraphQLResolveInfo): any;
/**
 * Create a GraphQL directive type definition.
 * @param directiveName Name of the directive
 */
export declare function createRateLimitTypeDef(directiveName?: string): any;
/**
 * Create an implementation of a rate limit directive.
 */
export declare function createRateLimitDirective<TContext>({ keyGenerator, pointsCalculator, onLimit, limiterClass, limiterOptions, }?: IOptions<TContext>): typeof SchemaDirectiveVisitor;
