"use strict";
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __spreadArray = (this && this.__spreadArray) || function (to, from, pack) {
    if (pack || arguments.length === 2) for (var i = 0, l = from.length, ar; i < l; i++) {
        if (ar || !(i in from)) {
            if (!ar) ar = Array.prototype.slice.call(from, 0, i);
            ar[i] = from[i];
        }
    }
    return to.concat(ar || Array.prototype.slice.call(from));
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableBuilder = void 0;
var table_config_class_1 = require("./table-config.class");
var table_snapshot_class_1 = require("./table-snapshot.class");
/**
 * Table Data API
 */
var TableBuilder = /** @class */ (function (_super) {
    __extends(TableBuilder, _super);
    /**
     * Create Table instance
     * @param columnHeaders optionally set columnHeaders
     */
    function TableBuilder() {
        var columnHeaders = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            columnHeaders[_i] = arguments[_i];
        }
        var _this = _super.call(this) || this;
        if (columnHeaders.length) {
            _this.setColumnHeaders(columnHeaders);
        }
        return _this;
    }
    Object.defineProperty(TableBuilder.prototype, "columns", {
        /**
         * Current number of columns (the value of the highest 'x' position of any cells + 1)
         */
        get: function () {
            return this._cells.columns;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TableBuilder.prototype, "rows", {
        /**
         * Current number of rows (the value of the highest 'y' position of any cells + 1)
         */
        get: function () {
            return this._cells.rows;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Append a row to the end of the table with the provided cell values (and possibly configuration)
     * @param values primitive value or { value, ...config }
     * @returns the Table
     */
    TableBuilder.prototype.addRow = function () {
        var values = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            values[_i] = arguments[_i];
        }
        return this.addRowWithHead.apply(this, __spreadArray([undefined], values, false));
    };
    /**
     * Append a row to the end of the table with the provided row header and cell values (and possibly configuration)
     * @param header primitive value or { value, ...config }
     * @param values primitive value or { value, ...config }
     * @returns the Table
     */
    TableBuilder.prototype.addRowWithHead = function (header) {
        var values = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            values[_i - 1] = arguments[_i];
        }
        return this.replaceRowWithHead.apply(this, __spreadArray([this.rows, header], values, false));
    };
    /**
     * Sets { borderBottom: true } on the last row's config
     * @returns this Table
     */
    TableBuilder.prototype.addSeparator = function () {
        return this.setRowConfig(this.rows - 1, { borderBottom: true });
    };
    /**
     * Clear the entire table, including values, configs, headers
     * @returns this Table
     */
    TableBuilder.prototype.clear = function () {
        this._cells.clear();
        this._cells.configs.clear();
        return this;
    };
    /**
     * Removes a column
     * @param x 0-based coordinate index (-1 for row headers)
     * @returns this Table
     */
    TableBuilder.prototype.deleteColumn = function (x) {
        return this.deleteColumns(x, 1);
    };
    /**
     * Removes a range of columns
     * @param x first column to delete: 0-based coordinate index (-1 for row headers)
     * @param count number of columns to remove
     * @returns this Table
     */
    TableBuilder.prototype.deleteColumns = function (x, count) {
        this._cells.splice(true, x, count);
        this._cells.configs.splice(true, x, count);
        return this;
    };
    /**
     * Removes a row
     * @param y 0-based coordinate index (-1 for column headers)
     * @returns this Table
     */
    TableBuilder.prototype.deleteRow = function (y) {
        return this.deleteRows(y, 1);
    };
    /**
     * Removes a range of rows
     * @param y first row to delete: 0-based coordinate index (-1 for column headers)
     * @param count number of rows to remove
     * @returns this Table
     */
    TableBuilder.prototype.deleteRows = function (y, count) {
        this._cells.splice(false, y, count);
        this._cells.configs.splice(false, y, count);
        return this;
    };
    /**
     * Flips x and y axes
     * @returns this Table
     */
    TableBuilder.prototype.flip = function () {
        this._cells.flip();
        this._cells.configs.flip();
        return this;
    };
    /**
     * Get cell value
     * @param x 0-based coordinate index (-1 for row header)
     * @param y 0 based coordinate index (-1 for column header)
     * @returns cell value
     */
    TableBuilder.prototype.getCell = function (x, y) {
        return this._cells.getsert(x, y).value;
    };
    /**
     * Get column headers
     * @returns undefined (for no headers) or array of header values
     */
    TableBuilder.prototype.getColumnHeaders = function () {
        return this._cells.headerValues(false);
    };
    /**
     * Get row headers
     * @returns undefined (for no headers) or array of header values
     */
    TableBuilder.prototype.getRowHeaders = function () {
        return this._cells.headerValues(true);
    };
    /**
     * A snapshot of the table's current state
     */
    TableBuilder.prototype.getSnapshot = function () {
        return new table_snapshot_class_1.TableSnapshot(this);
    };
    /**
     * Inserts a column
     * @param x 0 based coordinate index (-1 for column header)
     * @returns this Table
     */
    TableBuilder.prototype.insertColumn = function (x) {
        return this.insertColumns(x, 1);
    };
    /**
     * Inserts a number of columns
     * @param x 0 based coordinate index (-1 for column header)
     * @param count number of columns to insert
     * @returns this Table
     */
    TableBuilder.prototype.insertColumns = function (x, count) {
        this._cells.splice(true, x, 0, count);
        this._cells.configs.splice(true, x, 0, count);
        return this;
    };
    /**
     * Inserts a row, adds cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.insertRow = function (y) {
        var values = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            values[_i - 1] = arguments[_i];
        }
        this.insertRows(y, 1);
        return this.replaceRow.apply(this, __spreadArray([y], values, false));
    };
    /**
     * Inserts a row, adds head and cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param header primitive value or { value, ...config }
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.insertRowWithHead = function (y, header) {
        var values = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            values[_i - 2] = arguments[_i];
        }
        this.insertRows(y, 1);
        return this.replaceRowWithHead.apply(this, __spreadArray([y, header], values, false));
    };
    /**
     * Inserts a number of empty rows at coordinate
     * @param y first row to delete: 0-based coordinate index (-1 for column headers)
     * @param count number of rows to insert
     * @returns this Table
     */
    TableBuilder.prototype.insertRows = function (y, count) {
        this._cells.splice(false, y, 0, count);
        this._cells.configs.splice(false, y, 0, count);
        return this;
    };
    /**
     * Replaces a row's content with provided cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.replaceRow = function (y) {
        var values = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            values[_i - 1] = arguments[_i];
        }
        return this.replaceRowWithHead.apply(this, __spreadArray([y, undefined], values, false));
    };
    /**
     * Replaces a row's content with provided header and cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param header primitive value or { value, ...config }
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.replaceRowWithHead = function (y, header) {
        var values = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            values[_i - 2] = arguments[_i];
        }
        this._cells.remove(function (cell) { return cell.y === y; });
        this.setCell(-1, y, header);
        var len = values.length;
        for (var x = 0; x < len; x++) {
            this.setCell(x, y, values[x]);
        }
        return this;
    };
    /**
     * Update cell value (and optionally: config)
     * @param x 0-based coordinate index (-1 for row header)
     * @param y 0 based coordinate index (-1 for column header)
     * @param value new value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.setCell = function (x, y, value) {
        this._cells.upsert(x, y, value);
        return this;
    };
    /**
     * Update column headers (pass [] or null to delete headers)
     * @param newHeaders primitive value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.setColumnHeaders = function (newHeaders) {
        this._cells.headersReplace(false, newHeaders);
        return this;
    };
    /**
     * Update column headers (pass [] or null to delete headers)
     * @param newHeaders primitive value or { value, ...config }
     * @returns this Table
     */
    TableBuilder.prototype.setRowHeaders = function (newHeaders) {
        this._cells.headersReplace(true, newHeaders);
        return this;
    };
    return TableBuilder;
}(table_config_class_1.TableConfig));
exports.TableBuilder = TableBuilder;
