import { LambdaQueryFilter } from './lambda-query-filter';
import { FieldName } from './query-builder';
export declare enum Logical {
    and = "and",
    or = "or",
    not = "not"
}
export declare type Scalar = string | number | boolean | Date;
export declare enum GeoComparison {
    gt = "gt",
    lt = "lt",
    ge = "ge",
    le = "le"
}
/** Construct a filter string to be used with Azure Search */
export declare class QueryFilter<TDocument = any> {
    private mode;
    /** join multiple filters together with a logical NOT */
    static not<TDocument>(...filters: Array<QueryFilter<TDocument>>): QueryFilter<TDocument>;
    /** join multiple filters together with a logical OR */
    static or<TDocument>(...filters: Array<QueryFilter<TDocument>>): QueryFilter<TDocument>;
    /** join multiple filters together with a logical AND */
    static and<TDocument>(...filters: Array<QueryFilter<TDocument>>): QueryFilter<TDocument>;
    private static join;
    private expressions;
    constructor(mode?: Logical);
    /** append a new filter to this query using a logical AND */
    and(...filters: Array<QueryFilter<TDocument>>): this;
    /** append a new filter to this query using a logical OR */
    or(...filters: Array<QueryFilter<TDocument>>): this;
    /** append a new filter to this query using a logical NOT */
    not(...filters: Array<QueryFilter<TDocument>>): this;
    /** apply the equals operator */
    eq<K extends FieldName<TDocument>>(field: K, value: TDocument[K]): this;
    /** apply the not-equal operator */
    ne<K extends FieldName<TDocument>>(field: K, value: TDocument[K]): this;
    /** apply the greater-than operator */
    gt<K extends FieldName<TDocument>>(field: K, value: TDocument[K]): this;
    /** apply the less-than operator */
    lt<K extends FieldName<TDocument>>(field: K, value: TDocument[K]): this;
    /** apply the greater-than-or-equal-to operator */
    ge<K extends FieldName<TDocument>>(field: K, value: TDocument[K]): this;
    /** apply the less-than-or-equal-to operator */
    le<K extends FieldName<TDocument>>(field: K, value: TDocument[K]): this;
    /** apply the search.in filter */
    in<K extends FieldName<TDocument>>(field: K, values: string[], separator?: string): this;
    /**
     * Evaluates search query as a part of a filter expression
     * @param search the search query (in either simple or full query syntax).
     * @param searchFields searchable fields to search in, defaults to all searchable fields in the index.
     * @param queryType "simple" or "full", defaults to "simple". Specifies what query language was used in the search parameter.
     * @param searchMode "any" or "all", defaults to "any". Indicates whether any or all of the search terms must be matched in order to count the document as a match.
     */
    isMatch(search: string, searchFields?: Array<FieldName<TDocument>>, queryType?: 'simple' | 'full', searchMode?: 'any' | 'all'): this;
    /**
     * Evaluates search query as a part of a filter expression (with relevance score of documents matching this filter contributing to the overall document score)
     * @param search the search query (in either simple or full query syntax).
     * @param searchFields searchable fields to search in, defaults to all searchable fields in the index.
     * @param queryType "simple" or "full", defaults to "simple". Specifies what query language was used in the search parameter.
     * @param searchMode "any" or "all", defaults to "any". Indicates whether any or all of the search terms must be matched in order to count the document as a match.
     */
    isMatchScoring(search: string, searchFields?: Array<FieldName<TDocument>>, queryType?: 'simple' | 'full', searchMode?: 'any' | 'all'): this;
    /**
     * Filter string collection such that any member may match a predicate
     * @param field index field name
     * @param filter optional lambda filter. If omitted, the filter will return true if the collection has at least 1 item.
     */
    any<K extends FieldName<TDocument>>(field: K, filter?: (lambda: LambdaQueryFilter) => LambdaQueryFilter): this;
    /**
     * Filter string collection such that all members must match a predicate
     * @param field index field name
     * @param filter optional lambda filter. If omitted, the filter will return true if the collection has at least 1 item.
     */
    all<K extends FieldName<TDocument>>(field: K, filter?: (lambda: LambdaQueryFilter) => LambdaQueryFilter): this;
    /** apply a field reference filter */
    field(fieldName: FieldName<TDocument>): this;
    /**
     * Returns the distance in kilometers between two points, one being a field and one being a constant passed as part of the filter.
     * @param field index field name
     * @param point [lat, lon]
     * @param op comparison operator
     * @param value comparison operand
     */
    geoDistance<K extends FieldName<TDocument>>(field: K, point: [number, number], op: GeoComparison, value: number): this;
    /**
     * Returns true if a given point is within a given polygon, where the point is a field and the polygon is specified as a constant passed as part of the filter.
     * @param field index field name
     * @param polygon array of [lat, lon]
     */
    geoIntersects<K extends FieldName<TDocument>>(field: K, polygon: Array<[number, number]>): this;
    /** return filter as a string */
    toString(): string;
    private compare;
    private append;
    private prepValue;
    private isUnary;
}
