import { CellValue, CellValueOrValueWithOptions } from './cell/cell-value.type';
import { TableConfig } from './table-config.class';
import { TableSnapshot } from './table-snapshot.class';
/**
 * Table Data API
 */
export declare class TableBuilder extends TableConfig {
    /**
     * Create Table instance
     * @param columnHeaders optionally set columnHeaders
     */
    constructor(...columnHeaders: CellValueOrValueWithOptions[]);
    /**
     * Current number of columns (the value of the highest 'x' position of any cells + 1)
     */
    get columns(): number;
    /**
     * Current number of rows (the value of the highest 'y' position of any cells + 1)
     */
    get rows(): number;
    /**
     * Append a row to the end of the table with the provided cell values (and possibly configuration)
     * @param values primitive value or { value, ...config }
     * @returns the Table
     */
    addRow(...values: CellValueOrValueWithOptions[]): this;
    /**
     * Append a row to the end of the table with the provided row header and cell values (and possibly configuration)
     * @param header primitive value or { value, ...config }
     * @param values primitive value or { value, ...config }
     * @returns the Table
     */
    addRowWithHead(header?: CellValueOrValueWithOptions, ...values: CellValueOrValueWithOptions[]): this;
    /**
     * Sets { borderBottom: true } on the last row's config
     * @returns this Table
     */
    addSeparator(): this;
    /**
     * Clear the entire table, including values, configs, headers
     * @returns this Table
     */
    clear(): this;
    /**
     * Removes a column
     * @param x 0-based coordinate index (-1 for row headers)
     * @returns this Table
     */
    deleteColumn(x: number): this;
    /**
     * Removes a range of columns
     * @param x first column to delete: 0-based coordinate index (-1 for row headers)
     * @param count number of columns to remove
     * @returns this Table
     */
    deleteColumns(x: number, count: number): this;
    /**
     * Removes a row
     * @param y 0-based coordinate index (-1 for column headers)
     * @returns this Table
     */
    deleteRow(y: number): this;
    /**
     * Removes a range of rows
     * @param y first row to delete: 0-based coordinate index (-1 for column headers)
     * @param count number of rows to remove
     * @returns this Table
     */
    deleteRows(y: number, count: number): this;
    /**
     * Flips x and y axes
     * @returns this Table
     */
    flip(): this;
    /**
     * Get cell value
     * @param x 0-based coordinate index (-1 for row header)
     * @param y 0 based coordinate index (-1 for column header)
     * @returns cell value
     */
    getCell(x: number, y: number): CellValue;
    /**
     * Get column headers
     * @returns undefined (for no headers) or array of header values
     */
    getColumnHeaders(): CellValue[] | undefined;
    /**
     * Get row headers
     * @returns undefined (for no headers) or array of header values
     */
    getRowHeaders(): CellValue[] | undefined;
    /**
     * A snapshot of the table's current state
     */
    getSnapshot(): TableSnapshot;
    /**
     * Inserts a column
     * @param x 0 based coordinate index (-1 for column header)
     * @returns this Table
     */
    insertColumn(x: number): this;
    /**
     * Inserts a number of columns
     * @param x 0 based coordinate index (-1 for column header)
     * @param count number of columns to insert
     * @returns this Table
     */
    insertColumns(x: number, count: number): this;
    /**
     * Inserts a row, adds cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    insertRow(y: number, ...values: CellValueOrValueWithOptions[]): this;
    /**
     * Inserts a row, adds head and cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param header primitive value or { value, ...config }
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    insertRowWithHead(y: number, header: CellValueOrValueWithOptions, ...values: CellValueOrValueWithOptions[]): this;
    /**
     * Inserts a number of empty rows at coordinate
     * @param y first row to delete: 0-based coordinate index (-1 for column headers)
     * @param count number of rows to insert
     * @returns this Table
     */
    insertRows(y: number, count: number): this;
    /**
     * Replaces a row's content with provided cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    replaceRow(y: number, ...values: CellValueOrValueWithOptions[]): this;
    /**
     * Replaces a row's content with provided header and cell values
     * @param y 0 based coordinate index (-1 for column header)
     * @param header primitive value or { value, ...config }
     * @param values primitive value or { value, ...config }
     * @returns this Table
     */
    replaceRowWithHead(y: number, header: CellValueOrValueWithOptions, ...values: CellValueOrValueWithOptions[]): this;
    /**
     * Update cell value (and optionally: config)
     * @param x 0-based coordinate index (-1 for row header)
     * @param y 0 based coordinate index (-1 for column header)
     * @param value new value or { value, ...config }
     * @returns this Table
     */
    setCell(x: number, y: number, value: CellValueOrValueWithOptions): this;
    /**
     * Update column headers (pass [] or null to delete headers)
     * @param newHeaders primitive value or { value, ...config }
     * @returns this Table
     */
    setColumnHeaders(newHeaders: CellValueOrValueWithOptions[] | null | undefined): this;
    /**
     * Update column headers (pass [] or null to delete headers)
     * @param newHeaders primitive value or { value, ...config }
     * @returns this Table
     */
    setRowHeaders(newHeaders: CellValueOrValueWithOptions[] | null | undefined): this;
}
